/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#pragma once

#if MAX_RELEASE<13900
#include "maxscrpt/MAXScrpt.h"
#else
#include "maxscript/maxscript.h"
#endif

#include "field.h"

#include <algorithm>
#include <vector>

/*
	In static version there is no need to calculate point positions
	for each ray. So for container we'll just use the leaf.
*/
class BVHContainer {
public:
	virtual void getFieldNoTest(Field* field) = 0;
};

/*
	When we trace the BVH each leaf thats along the ray is turned into these
	They contain the positions along the ray where it enters the bounding box
	of the leaf and when it exits it. This means the BVH has to be intersected
	for each ray only once.
*/
class BVHStep {
public:
	VR::Ireal min;
	VR::Ireal max;
	BVHContainer* cont;
	BVHStep* next;
	
	BVHStep(VR::Ireal min, VR::Ireal max, BVHContainer* cont, BVHStep* next) {
		this->min=min; this->max=max; this->cont=cont; this->next=next;
	}
	~BVHStep() {delete next;}
	void deleteThisOnly() {next = NULL; delete this;} // Used when optimizing, sometimes we want to remove individual steps
};
struct BVHStepSort {
     bool operator()(BVHStep*& a, BVHStep*& b) {
          return a->min < b->min;
     }
};

/*
	Ray structure that contains all the leafs along single ray
*/
class BVHRay {	
public:
	BVHStep* first;	
	std::vector<BVHStep*> steps;

	void add(VR::Ireal min, VR::Ireal max, BVHContainer* cont);	
	void sortSteps();
	void collapseSteps();

	BVHRay() {first = NULL; steps.reserve(25); }
	~BVHRay() {delete first;}
};

/*
	Parent class for BVH nodes
*/
class BVHNode {
public:
	VR::Box b;
		
	BVHNode() {}
	~BVHNode() {}

	int intersect(VR::IRay* ray);

	// Override
	virtual int type() {return 0;}
	virtual void traceRay(VR::IRay* ray, BVHRay* bRay) = 0;
	virtual void getField(Field* field) = 0;
	virtual void getGradValue(FieldGrad* grad) = 0;

	// For branches
	virtual BVHNode* getLeft() {return NULL;}
	virtual BVHNode* getRight() {return NULL;}

	// For leaves
	virtual int* getPoints() {return NULL;}	
	virtual int getCount() {return 0;}
};

/*
	BVH branch class
*/
class Branch: public BVHNode {
public:
	BVHNode* left;
	BVHNode* right;

	int type() {return 1;}			
	void traceRay(VR::IRay* ray, BVHRay* bRay);
	void getField(Field* field);
	void getGradValue(FieldGrad* grad);

	BVHNode* getLeft() {return left;}
	BVHNode* getRight() {return right;}

	Branch(VR::Vector* pointCloud, VR::Ireal* radii, int* points, int count, VR::real size, int maxDepth, int depth, int leafSize, VR::real maxLength);	
	~Branch() {
		delete left;
		delete right;
	}
};


/*
	BVH leaf class, leafs only store indexes of points, they don't
	store the actual points.
*/
class Leaf: public BVHNode, public BVHContainer {
public:
	int* points;
	int count;

	int type() {return 0;}	
	void traceRay(VR::IRay* ray, BVHRay* bRay);
	void getField(Field* field);
	void getFieldNoTest(Field* field);
	void getGradValue(FieldGrad* grad);

	int* getPoints() {return points;}
	int getCount() {return count;}	
	
	Leaf(VR::Vector* pointCloud, VR::Ireal* radii, int* points, int count, VR::real size);	
	~Leaf() {
		delete[] points;
	}
};

/*
	BVH base class
*/
class BoundingVolumeHierarchy {
private:
	BVHNode* root;
	
public:
	BoundingVolumeHierarchy() {
		root = NULL;
	}

	~BoundingVolumeHierarchy() {
		delete root;
	}

	void create(VR::Vector* pointCloud, VR::Ireal* radii, int count, VR::real size, int maxDepth, int leafSize, VR::real maxLength);
	int intersect(VR::IRay* ray) {	
		return root->intersect(ray);
	}
	void traceRay(VR::IRay* ray, BVHRay* bRay) {
		root->traceRay(ray, bRay);
		bRay->sortSteps();
		bRay->collapseSteps();			
	}
	void getField(Field* field) {
		root->getField(field);
	}
	void getGradValue(FieldGrad* grad) {
		root->getGradValue(grad);
	}
};