/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#include "instanceStatic.h"
#include "primitive.h"
#include "tomax.h"
#include "misc_ray.h"

#if MAX_RELEASE<13900
#include "maxscrpt/MAXScrpt.h"
#else
#include "maxscript/maxscript.h"
#endif

#include "readPointCloud.h"

using namespace VR;

BerconMetaballInstanceStatic::BerconMetaballInstanceStatic(BerconMetaball *META, INode *node, VRayCore *vray, int renderID) {
	dynamicType = 0;
	VRenderInstance::init(META, node, vray, renderID);
}

BerconMetaballInstanceStatic::~BerconMetaballInstanceStatic(void) {
}

void BerconMetaballInstanceStatic::renderBegin(TimeValue t, VRayCore *vray) {
	VRenderInstance::renderBegin(t, vray);
	raycache.renderBegin((VR::VRayRenderer*) vray);
}

void BerconMetaballInstanceStatic::renderEnd(VRayCore *vray) {
	VRenderInstance::renderEnd(vray);
	raycache.renderEnd((VR::VRayRenderer*) vray);
}

void BerconMetaballInstanceStatic::frameBegin(TimeValue t, VRayCore *vray) {
	VRenderInstance::frameBegin(t, vray);

	objToWorld=node->GetObjTMAfterWSM(t);
	Matrix3 worldToObj=Inverse(objToWorld);

	camToObj=toMatrix3(toTransform(worldToObj)*VR::Transform(vray->getFrameData().camToWorld));
	objToCam=toMatrix3(VR::Transform(vray->getFrameData().worldToCam)*toTransform(objToWorld));

	tm=toTransform(objToWorld);
	itm=inverse(tm);

	mesh=&dummyMesh;

	BerconMetaball *maxObj=((BerconMetaball*) renderObject);

	MetaballParams params;	
	params.size			= maxObj->pblock2->GetFloat(pb_size, t);
	params.threshold	= maxObj->pblock2->GetFloat(pb_threshold, t);
	params.step			= maxObj->pblock2->GetFloat(pb_step, t);
	params.error		= maxObj->pblock2->GetFloat(pb_error, t);
	params.field		= maxObj->pblock2->GetInt(pb_fieldFunction, t);
	params.color		= maxObj->pblock2->GetInt(pb_color, t);
	params.useAverage	= (bool)maxObj->pblock2->GetInt(pb_useAverage, t);
	params.cutoff		= maxObj->pblock2->GetFloat(pb_cutoff, t);
	params.trim			= maxObj->pblock2->GetFloat(pb_trim, t);
	params.depth		= maxObj->pblock2->GetInt(pb_depth, t);	
	params.leafLength	= maxObj->pblock2->GetFloat(pb_leafLength, t);
	params.leafSize		= maxObj->pblock2->GetInt(pb_leafSize, t);
	params.texStr		= maxObj->pblock2->GetFloat(pb_texstr, t);
	params.texType		= (bool)maxObj->pblock2->GetInt(pb_textype, t);
	params.clamp		= (bool)maxObj->pblock2->GetInt(pb_clamp, t);
	params.clampTo		= maxObj->pblock2->GetFloat(pb_clampTo, t);	
	params.tex = NULL;
	if (maxObj->pblock2->GetInt(pb_useTexture, t) && params.texStr > 0.0001f) {		
		maxObj->pblock2->GetValue(pb_texmap, t, params.tex, Interval(0,0));
		if (params.tex)
			params.tex->Update(t,Interval(0,0));	
	}
	bool useRadii		= (bool)maxObj->pblock2->GetInt(pb_useScale, t);

	// Check that all parameters are valid
	if (params.clampTo <= params.threshold) params.clampTo = params.threshold + 0.001f;	

	int pointCount;
	Vector *points = NULL;
	Ireal *sizes = NULL; // This remains NULL if we don't use radii
	Vector *colors = NULL;

	readPointCloud reader;
	reader.defaultRadius = params.size;
	reader.read(useRadii, params.color, pointCount, points, sizes, colors, maxObj->pblock2, t);

	if (params.color == 1) params.color = params.field;
	else params.color -= 2;
	
	raycache.frameBegin((VR::VRayRenderer*) vray);

	META.init(params, points, sizes, colors, pointCount, (GeometryGenerator*) this, ownerIndex, &raycache);
}

void BerconMetaballInstanceStatic::frameEnd(VRayCore *vray) {
	VRenderInstance::frameEnd(vray);
	raycache.frameEnd((VR::VRayRenderer*) vray);
}

void BerconMetaballInstanceStatic::compileGeometry(VRayCore *vray) {
	VR::RayServerInterface *rayserver=vray->getRayServer();
	firstPrimID=-1;
	rayserver->storeStaticPrimitive(static_cast<StaticPrimitive*>(this), static_cast<GeometryGenerator*>(this), rayserver->getNewRenderID());
}

void BerconMetaballInstanceStatic::setIntersectionData(RSRay &rsray, void *isd) {
	IntersectionData &isData=*((IntersectionData*) isd);
	RSIntersection &is=rsray.is;
	BerconMetaballPrimitiveStatic *prim=(BerconMetaballPrimitiveStatic*) rsray.is.primitive;
	isData.primitive=is.primitive;
	isData.sb=(Shadeable*) this;
	isData.sd=(ShadeData*) this;
	isData.si=(ShadeInstance*) this;
	isData.surfaceProps=&surfaceProps;
	isData.volume=NULL;
	isData.skipTag=rsray.is.skipTag;
	isData.faceIndex=0;
	
	SurfaceInfo s;
	prim->getSurfaceInfo(&s, rsray);
	isData.bary = s.col;
	//isData.bary.set(0.0f, 0.0f, 0.0f);	

	isData.wpoint=rsray.p+TracePoint(rsray.dir)*is.t;
	isData.wpointCoeff=is.t;
	isData.gnormal=prim->getGNormal(rsray);
	isData.normal=isData.gnormal;
}

/*
void BerconMetaballInstanceStatic::setIntersectionData(RSRayRef& rsrayref, void *isd) {
	IntersectionData &isData=*((IntersectionData*) isd);
	BerconMetaballPrimitiveStatic *prim=(BerconMetaballPrimitiveStatic*) rsrayref.getGenericPrimitive();
	isData.primitive = rsrayref.getGenericPrimitive();
	isData.sb=(Shadeable*) this;
	isData.sd=(ShadeData*) this;
	isData.si=(ShadeInstance*) this;
	isData.surfaceProps=&surfaceProps;
	isData.volume=NULL;
	//isData.skipTag = rsrayref.getSkipTag();
	isData.faceIndex=0;
	
	SurfaceInfo s;
	prim->getSurfaceInfo(&s, rsrayref);
	isData.bary = s.col;
	//isData.bary.set(0.0f, 0.0f, 0.0f);	

	isData.wpoint= rsrayref.getOrigin() + TracePoint(rsrayref.getDirection()) * rsrayref.getDistance();
	isData.wpointCoeff = rsrayref.getDistance();
	isData.gnormal= prim->getGNormal(rsrayref);
	isData.normal= isData.gnormal;
}
*/

// From RenderInstance
Interval BerconMetaballInstanceStatic::MeshValidity(void) { return FOREVER; }
Point3 BerconMetaballInstanceStatic::GetFaceNormal(int faceNum) { return Point3(0,0,1); }
Point3 BerconMetaballInstanceStatic::GetFaceVertNormal(int faceNum, int vertNum) { return Point3(0,0,1); }
void BerconMetaballInstanceStatic::GetFaceVertNormals(int faceNum, Point3 n[3]) {}
Point3 BerconMetaballInstanceStatic::GetCamVert(int vertNum) { return Point3(0,0,0); }
void BerconMetaballInstanceStatic::GetObjVerts(int fnum, Point3 obp[3]) {}
void BerconMetaballInstanceStatic::GetCamVerts(int fnum, Point3 cp[3]) {}

// From Shadeable
void BerconMetaballInstanceStatic::shade(VR::VRayContext &vri) {VRenderInstance::fullShade((VR::VRayInterface&) vri);}

// From ShadeData
VR::Vector BerconMetaballInstanceStatic::getUVWcoords(const VR::VRayContext &vri, int channel) {return VR::Vector(0,0,0);}
void BerconMetaballInstanceStatic::getUVWderivs(const VR::VRayContext &vri, int channel, VR::Vector derivs[2]) {
		derivs[0]=normalize(tm.m[0]);derivs[1]=normalize(tm.m[1]);}		
void BerconMetaballInstanceStatic::getUVWbases(const VR::VRayContext &vri, int channel, VR::Vector bases[3]) {
		bases[0]=normalize(tm.m[0]);bases[1]=normalize(tm.m[1]);bases[2]=normalize(tm.m[2]);}
VR::Vector BerconMetaballInstanceStatic::getUVWnormal(const VR::VRayContext &vri, int channel) {return normalize(tm.m[2]);}
int BerconMetaballInstanceStatic::getMtlID(const VR::VRayContext &vri) { return 0; }
int BerconMetaballInstanceStatic::getGBufID(void) { return 0; }
int BerconMetaballInstanceStatic::getSurfaceRenderID(const VR::VRayContext &rc) { return VRenderInstance::renderID; }
int BerconMetaballInstanceStatic::getMaterialRenderID(const VR::VRayContext &rc) { return 0; }
void BerconMetaballInstanceStatic::getSmoothUVWbases(const VR::VRayContext &vri, int channel, VR::Vector bases[3]) {getUVWbases(vri, channel, bases);}

// From ShadeInstance
VR::Vector BerconMetaballInstanceStatic::getBasePt(const VR::VRayContext &vri, BaseTime baseTime) { return vri.rayresult.wpoint; }
VR::Vector BerconMetaballInstanceStatic::getBasePtObj(const VR::VRayContext &vri) { return vri.rayresult.wpoint; }
VR::Vector BerconMetaballInstanceStatic::worldToObjectVec(const VR::VRayContext &vri, const VR::Vector &d) { return d; }
VR::Vector BerconMetaballInstanceStatic::worldToObjectPt(const VR::VRayContext &vri, const VR::Vector &p) { return p; }
VR::Vector BerconMetaballInstanceStatic::objectToWorldVec(const VR::VRayContext &vri, const VR::Vector &d) { return d; }
VR::Vector BerconMetaballInstanceStatic::objectToWorldPt(const VR::VRayContext &vri, const VR::Vector &p) { return p; }
VR::Vector BerconMetaballInstanceStatic::getBaseGNormal(const VR::VRayContext &vri, BaseTime baseTime) { return vri.rayresult.gnormal; }
VR::Vector BerconMetaballInstanceStatic::getBaseNormal(const VR::VRayContext &vri, BaseTime baseTime) { return vri.rayresult.origNormal; }
VR::Vector BerconMetaballInstanceStatic::getBaseBumpNormal(const VR::VRayContext &vri, BaseTime baseTime) { return vri.rayresult.normal; }
VR::TracePoint BerconMetaballInstanceStatic::getShadowPt(const VR::VRayContext &vri) { return vri.rayresult.wpoint; }
VR::Vector BerconMetaballInstanceStatic::getVelocity(const VR::VRayContext &vri) { return VR::Vector(0,0,0); }

// From DynamicPrimitive
int BerconMetaballInstanceStatic::expandable(void) { return true; }
bool BerconMetaballInstanceStatic::splittable(void) { return true; }