/*
Licensed to the Apache Software Foundation (ASF) under one
or more contributor license agreements. The ASF licenses this
file to you under the Apache License, Version 2.0 (the
"License"); you may not use this file except in compliance
with the License.  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing,
software distributed under the License is distributed on an
"AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
KIND, either express or implied.  See the License for the
specific language governing permissions and limitations
under the License.   
*/

#include "readPointCloud.h"

#include "ParticleFlow/PFExport.h"
#include "ParticleFlow/PFClassIDs.h"
#include "ParticleFlow/IParticleGroup.h"
#include "ParticleFlow/IParticleContainer.h"
#include "ParticleFlow/IParticleChannelPoint3.h"
#include "ParticleFlow/IParticleChannelPosition.h"
#include "ParticleFlow/IParticleChannelScale.h"
#include "ParticleFlow/IParticleChannelMXSVector.h"
#include "ParticleFlow/IParticleChannelID.h"

static int compareParticleID(const void* particle1, const void* particle2) {
	return ((Particle*)particle1)->id - ((Particle*)particle2)->id;
}

/*
	This is modified from pflow if I remember correctly, basicly it turns any
	object into mesh thats easy to handle if possible.
*/
static TriObject* GetTriObjectFromNode(INode *iNode, const TimeValue t, bool &deleteIt) {
	deleteIt = false;
	if (iNode == NULL) return NULL;
	Object *obj = iNode->EvalWorldState(t).obj;	
	if (obj == NULL) return NULL;
    if (obj->SuperClassID()==GEOMOBJECT_CLASS_ID) {
		if (obj->IsSubClassOf(triObjectClassID)) {
		  return (TriObject*)obj;
		} else if (obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0))) { 
			TriObject *tri = (TriObject *) obj->ConvertToType(t, Class_ID(TRIOBJ_CLASS_ID, 0));
			if (obj != tri) deleteIt = true;
			return tri;
		} else {
			return NULL;
		}
	} else {
		return NULL;
	}
}

/*
	Reads vertices from mesh as point cloud
*/

void readPointCloud::readVertexColors(Mesh theMesh) {
	/*
		Each face can have 3 color vertex so there can be huge number of different
		vertex colors per actual vertex. This algorithm simply reads one randomly
		out of those and uses it. Usually the colors for single vertex are same
		anyway. There is no "correct" way of doing this so this is as good as any.
	*/
	
	int numVertCol = theMesh.getNumVertCol();
	int numFaces = theMesh.getNumFaces();
	
	if(!numVertCol || !theMesh.vertCol || !theMesh.vcFace) return; // No vertex color information available

	Point3* colors = new Point3[theMesh.numVerts]; // Temporary storage for collected colors

	for (int i=0; i<numFaces; i++) 	{
		TVFace vcFace = theMesh.vcFace[i];
		Face vFace = theMesh.faces[i];		
		for (int j=0; j<3; j++) { // For each COLOR vertex
			colors[vFace.v[j]] = theMesh.vertCol[vcFace.t[j]];
		}
	}

	CharStream *out = thread_local(current_stdout);
				
	for (int i=0; i<theMesh.numVerts; i++) { // Append to color tab
		colTab.Append(1,&VR::Vector(colors[i].x, colors[i].y, colors[i].z),500);		
	}

	delete[] colors;
}

bool readPointCloud::readTriMesh(INode *node, TimeValue time) {
	bool deleteIt;
	TriObject *triObj = GetTriObjectFromNode(node, time, deleteIt);	
	if (triObj) { // Meshable, lets add the points
		Mesh theMesh = triObj->GetMesh();
		Matrix3 tm = node->GetObjectTM(time);
		Point3* pos;

		float* softSel;
		float vWeight;
		if (useRadii)
			 softSel = theMesh.getVSelectionWeights();


		for (int i=0; i<theMesh.numVerts; i++) {
			pos = &tm.PointTransform(theMesh.verts[i]);
			posTab.Append(1,&VR::Vector(pos->x, pos->y, pos->z),500);
			if (useRadii) {
				if (softSel) {
					vWeight = defaultRadius * softSel[i];
					radTab.Append(1,&vWeight,500);
				} else {
					radTab.Append(1,&defaultRadius,500);
				}
			}
		}			

		if (useColors) readVertexColors(theMesh);

		if (deleteIt) triObj->DeleteMe();			
		return true;
	}
	return false; // Not valid tri mesh
}

bool readPointCloud::readTriMesh(INode *node, TimeValue time[2]) {	
	bool deleteIt;
	bool deleteIt2;
	TriObject *triObj = GetTriObjectFromNode(node, time[0], deleteIt);		
	if (triObj) { // Meshable, lets add the points				
		Mesh theMesh = triObj->GetMesh();
		if (deleteIt) triObj->DeleteMe();
		TriObject *triObj2 = GetTriObjectFromNode(node, time[1], deleteIt2);	
		Mesh theMesh2 = triObj2->GetMesh();			
		if (deleteIt2) triObj2->DeleteMe();
		Matrix3 tm = node->GetObjectTM(time[0]);
		Matrix3 tm2 = node->GetObjectTM(time[1]);
		Point3 pos;
		Point3 pos2;	

		float* softSel;
		float* softSel2;
		float vWeight;
		float vWeight2;
		if (useRadii) {
			softSel = theMesh.getVSelectionWeights();
			softSel2 = theMesh2.getVSelectionWeights();
		}

		if (theMesh.numVerts != theMesh2.numVerts)
			return false; // Changing vertex count, we can't compute velocities

		for (int i=0; i<theMesh.numVerts; i++) {
			pos = tm.PointTransform(theMesh.verts[i]);
			pos2 = tm2.PointTransform(theMesh2.verts[i]) - pos;
			posTab.Append(1,&VR::Vector(pos.x, pos.y, pos.z),500);			
			velTab.Append(1,&VR::Vector(pos2.x, pos2.y, pos2.z),500);
			if (useRadii) {
				if (softSel && softSel2) {
					vWeight = defaultRadius * softSel[i];
					vWeight2 = defaultRadius * softSel2[i] - vWeight; // This is the delta
					radTab.Append(1,&vWeight,500);
					chaTab.Append(1,&vWeight2,500);
				} else {
					radTab.Append(1,&defaultRadius,500);
					chaTab.Append(1,&zero,500);
				}
			}
		}						

		if (useColors) readVertexColors(theMesh);

		return true;
	}
	return false; // Not valid tri mesh
}

/*
	Reads simple particle system
*/
bool readPointCloud::readSimpleParticle(INode *node, TimeValue time[2]) {
	ObjectState tos =  node->EvalWorldState(time[0],TRUE);
	SimpleParticle *pobj = (SimpleParticle*) tos.obj->GetInterface(I_SIMPLEPARTICLEOBJ);
	if (pobj) { // Simple particle system
		pobj->UpdateParticles(time[0], node);
		int count = pobj->parts.Count();
		for (int pid = 0; pid < count; pid++) {
			TimeValue age  = pobj->ParticleAge(time[0],pid);
			if (age!=-1) {
				Point3* pos;
				pos = &pobj->parts.points[pid];			
				posTab.Append(1,&VR::Vector(pos->x, pos->y, pos->z),500);
				if (useColors) colTab.Append(1,&VR::Vector(0,0,0),500);
				Point3* vel;
				vel = &pobj->parts.vels[pid];			
				velTab.Append(1,&(VR::Vector(vel->x, vel->y, vel->z) * (float)(time[1]-time[0])),500);				
				if (useRadii) {
					float rad;
					if (pobj->parts.radius.Count()) rad = pobj->parts.radius[pid];
					radTab.Append(1,&rad,500);
					chaTab.Append(1,&zero,500); // TODO: Evaluate particles at time[1] to get the change in radius					
				}
			}
		}
		return true;
	}
	return false;
}

bool readPointCloud::readSimpleParticle(INode *node, TimeValue time) {
	ObjectState tos =  node->EvalWorldState(time,TRUE);
	SimpleParticle *pobj = (SimpleParticle*) tos.obj->GetInterface(I_SIMPLEPARTICLEOBJ);
	if (pobj) { // Simple particle system
		pobj->UpdateParticles(time, node);
		int count = pobj->parts.Count();
		for (int pid = 0; pid < count; pid++) {
			TimeValue age  = pobj->ParticleAge(time,pid);
			if (age!=-1) {
				Point3* pos;
				pos = &pobj->parts.points[pid];			
				posTab.Append(1,&VR::Vector(pos->x, pos->y, pos->z),500);
				if (useColors) colTab.Append(1,&VR::Vector(0,0,0),500);
				if (useRadii) {
					float rad;
					if (pobj->parts.radius.Count()) rad = pobj->parts.radius[pid];
					radTab.Append(1,&rad,500);				
				}
			}
		}
		return true;
	}
	return false;
}

/*
	Reads extended particle system
*/
static void collectParticleData(IParticleObjectExt* epobj, Tab<INode *> &pfList, Tab<Particle> &atTime, BOOL useAllPFEvents, bool useRadii, bool useColors, TimeValue t) {
	int count = epobj->NumParticles();
	Tab<INode*> groups;

	for (int i=0; i<count; i++) { // TODO: Brute force search of groups, is there no better way of finding all groups?
		INode *group = epobj->GetParticleGroup(i);
		int j = 0;
		while (j < groups.Count()) {
			if (groups[j] == group)
				break;
			j++;
		}
		if (j >= groups.Count())
			groups.Append(1, &group);
	}

	CharStream *out = thread_local(current_stdout);	

	for (int i=0; i<groups.Count(); i++) {
		INode* group = groups[i];
		BOOL useGroup = TRUE;			
		if (!useAllPFEvents) {
			useGroup = FALSE;
			for (int j=0; j<pfList.Count(); j++)
				if (group == pfList[j]) {
					useGroup = TRUE;
					break;
				}
		}
		if (!useGroup) continue; // Skip

		IParticleGroup* groupIntf = ParticleGroupInterface(group->GetObjectRef());
		IObject* iObj = groupIntf->GetParticleContainer();
		IParticleContainer* partCont = (IParticleContainer*)iObj;
		
		//groupIntf->InvalidateCaches();
		groupIntf->Update(t);		
		
		IParticleChannelPoint3R* pChannelMXSVector = NULL;
		IParticleChannelPoint3R* pChannelScale = NULL;
		IParticleChannelPoint3R* pChannelPosition = NULL;
		IParticleChannelIDR* pChannelID = NULL;


		pChannelID = GetParticleChannelIDRInterface(iObj);
		if (!pChannelID) return;

		if (useColors) {
			pChannelMXSVector = (IParticleChannelPoint3R*) partCont->EnsureInterface(PARTICLECHANNELMXSVECTORR_INTERFACE,
																		ParticleChannelPoint3_Class_ID,
																		true, PARTICLECHANNELMXSVECTORR_INTERFACE,
																		PARTICLECHANNELMXSVECTORW_INTERFACE, true,
																		NULL, NULL, FALSE);
			if (!pChannelMXSVector)
				return;
		}

		if (useRadii) {
			pChannelScale = (IParticleChannelPoint3R*) partCont->EnsureInterface(PARTICLECHANNELSCALER_INTERFACE,
																		ParticleChannelPoint3_Class_ID,
																		true, PARTICLECHANNELSCALER_INTERFACE,
																		PARTICLECHANNELSCALEW_INTERFACE, true,
																		NULL, NULL, FALSE);
			if (!pChannelScale)
				return;
		}

		pChannelPosition = (IParticleChannelPoint3R*) partCont->EnsureInterface(PARTICLECHANNELPOSITIONR_INTERFACE,
																	ParticleChannelPoint3_Class_ID,
																	true, PARTICLECHANNELPOSITIONR_INTERFACE,
																	PARTICLECHANNELPOSITIONW_INTERFACE, true,
																	NULL, NULL, FALSE);
		if (!pChannelPosition)
			return;

		for (int pid=0; pid<partCont->Count(); pid++) {
			Particle p;
			p.id = pChannelID->GetParticleBorn(pid);//epobj->GetParticleBornIndex(pid);
			if (useColors) {
				Point3 col = pChannelMXSVector->GetValue(pid);
				p.color = VR::Vector(col.x, col.y, col.z);
			}		
			if (useRadii) {
				p.radius = pChannelScale->GetValue(pid).x; // Use x scale as size
			}
			Point3 pos = pChannelPosition->GetValue(pid);
			p.position = VR::Vector(pos.x, pos.y, pos.z);
			
			atTime.Append(1,&p,500);
		}
	}
}

bool readPointCloud::readExtParticle(INode *node, IParamBlock2 *pblock, TimeValue time) {
	ObjectState tos =  node->EvalWorldState(time,TRUE);
	IParticleObjectExt* epobj = (IParticleObjectExt*) tos.obj->GetInterface(PARTICLEOBJECTEXT_INTERFACE);
	if (epobj) {
		Tab<INode *> pfList;
		BOOL useAllPFEvents;
		pblock->GetValue(pb_allPflow,0,useAllPFEvents,FOREVER);
		int pfeventCount = pblock->Count(pb_pflow);
		for (int i = 0; i < pfeventCount; i++) {
			INode *pfevent;
			pblock->GetValue(pb_pflow, 0, pfevent, FOREVER, i);
			if (node)
				pfList.Append(1,&pfevent);
		}
		
		Tab<Particle> atTime0;		
		epobj->UpdateParticles(node, time);
		collectParticleData(epobj, pfList, atTime0, useAllPFEvents, useRadii, useColors, time);

		int count = atTime0.Count(); 
		for (int i=0; i<count; i++) {			
			posTab.Append(1,&(atTime0[i].position),500);
			if (useRadii)
				radTab.Append(1,&(atTime0[i].radius),500);							

			colTab.Append(1,&(atTime0[i].color),500);
		}

		atTime0.SetCount(0);
		return true;
	}
	return false;
}

/* Reads pos + velocity channel, obsolete and bad method.
bool readPointCloud::readExtParticleVel(INode *node, IParamBlock2 *pblock, TimeValue time[2]) {
	ObjectState tos =  node->EvalWorldState(time[0],TRUE);
	IParticleObjectExt* epobj = (IParticleObjectExt*) tos.obj->GetInterface(PARTICLEOBJECTEXT_INTERFACE);
	if (epobj) {
		Tab<INode *> pfList;
		BOOL useAllPFEvents;
		pblock->GetValue(pb_allPflow,0,useAllPFEvents,FOREVER);
		int pfeventCount = pblock->Count(pb_pflow);
		for (int i = 0; i < pfeventCount; i++) {
			INode *pfevent;
			pblock->GetValue(pb_pflow, 0, pfevent, FOREVER, i);
			if (node)
				pfList.Append(1,&pfevent);
		}

		epobj->UpdateParticles(node, time[0]);

		int count = epobj->NumParticles();
		Point3 noVel = Point3(0.f,0.f,0.f);
		for (int pid = 0; pid < count; pid++) {
			TimeValue age  = epobj->GetParticleAgeByIndex(pid);
			if (age==-1) continue; // Bad age skip particle
			Point3 *pos = epobj->GetParticlePositionByIndex(pid);
			float scale = epobj->GetParticleScaleByIndex(pid); 
			Point3 *vel = epobj->GetParticleSpeedByIndex(pid);			
			if (!vel) vel = &noVel;
			INode *group = epobj->GetParticleGroup(pid);
			BOOL useParticle = TRUE;			
			if (!useAllPFEvents) {
				useParticle = FALSE;
				for (int j = 0; j < pfList.Count(); j++)
					if (group == pfList[j]) {
						useParticle = TRUE;
						break;
					}
			}
			if ((pos) && (vel) && (useParticle)) {				
				VR::Vector position = VR::Vector(pos->x, pos->y, pos->z);			
				VR::Vector velocity = VR::Vector(vel->x, vel->y, vel->z) * (float)(time[1] - time[0]);
				posTab.Append(1,&position,500);								
				velTab.Append(1,&velocity,500);				
				radTab.Append(1,&(scale),500);
				chaTab.Append(1,&zero,500);
			}
		}
		return true;
	}
	return false;
}*/

bool readPointCloud::readExtParticle(INode *node, IParamBlock2 *pblock, TimeValue time[2]) {
	ObjectState tos =  node->EvalWorldState(time[0],TRUE);
	IParticleObjectExt* epobj = (IParticleObjectExt*) tos.obj->GetInterface(PARTICLEOBJECTEXT_INTERFACE);
	if (epobj) {
		Tab<INode *> pfList;
		BOOL useAllPFEvents;
		pblock->GetValue(pb_allPflow,0,useAllPFEvents,FOREVER);
		int pfeventCount = pblock->Count(pb_pflow);
		for (int i = 0; i < pfeventCount; i++) {
			INode *pfevent;
			pblock->GetValue(pb_pflow, 0, pfevent, FOREVER, i);
			if (node)
				pfList.Append(1,&pfevent);
		}

		// Collect particle data at the start and end of the time interval
		Tab<Particle> atTime0;		
		epobj->UpdateParticles(node, time[0]);
		collectParticleData(epobj, pfList, atTime0, useAllPFEvents, useRadii, useColors, time[0]);		

		Tab<Particle> atTime1;
		epobj->UpdateParticles(node, time[1]);
		collectParticleData(epobj, pfList, atTime1, useAllPFEvents, useRadii, false, time[1]);

		// Intersect two particle data states, only those particles that exists in both are used, those born and deleted within are ignored
		atTime0.Sort(compareParticleID);
		atTime1.Sort(compareParticleID);

		int count = atTime0.Count(); 
		int j = 0;
		for (int i=0; i<count; i++) {			
			while (atTime1[j].id < atTime0[i].id)
				j++;
			if (atTime0[i].id == atTime1[j].id) { // State at time 0 and time 1 both exists, append particle to our Tabs				
				VR::Vector velocity = atTime1[j].position-atTime0[i].position;
				if (velThresh < 0 || velocity.length() / (time[1] - time[0]) * 160 < velThresh) {
					posTab.Append(1,&(atTime0[i].position),500);
					if (useColors) colTab.Append(1,&(atTime0[i].color),500);
					
					velTab.Append(1,&velocity,500);				
					if (useRadii) {
						radTab.Append(1,&(atTime0[i].radius),500);
						float change = atTime1[j].radius-atTime0[i].radius;				
						chaTab.Append(1,&change,500);				
					}
				}
			}			
		}

		atTime0.SetCount(0);
		atTime1.SetCount(0);
		return true;
	}
	return false;
}	

/*
	Reads all nodes into points
*/

void readPointCloud::read(bool useRadii, bool useColors, int &pointCount, VR::Vector* &position, VR::Ireal* &radii, VR::Vector* &colors, IParamBlock2 *pblock, TimeValue time) {
	this->useRadii = useRadii;

	// Start parsing nodes
	int nodeCount = pblock->Count(pb_nodes);
	for (int i=0; i<nodeCount; i++) {
		INode *node;
		pblock->GetValue(pb_nodes,time,node,Interval(time,time),i);
		if (!node) continue; // No node skip

		// Mesh
		if (readTriMesh(node, time))
			continue;

		// Particles
		ObjectState tos =  node->EvalWorldState(time,TRUE);
		if (tos.obj->IsParticleSystem()) {				
			// Simple particle system
			if (readSimpleParticle(node, time))
				continue;	
			// Extended particle system
			if (readExtParticle(node, pblock, time))
				continue;	
		}		

		// TODO: Add support for more objects here
	}

	// Extra metaball points, the Point3 array which can be used through maxscript for example
	int count = pblock->Count(pb_points);
	int count2 = pblock->Count(pb_pointSizes);
	int count3 = pblock->Count(pb_colors);
	Interval ivalid = Interval(time,time);
	for (int i=0; i<count; i++) {
		Point3 pos;
		pblock->GetValue(pb_points,time,pos,ivalid,i);
		posTab.Append(1,&VR::Vector(pos.x, pos.y, pos.z),500);
		if (useRadii) {		
			if (i < count2) {
				float size;
				pblock->GetValue(pb_pointSizes,time,size,ivalid,i);
				radTab.Append(1,&size,500);
			} else {
				radTab.Append(1,&defaultRadius,500);
			}
		}			
		if (useColors) {		
			Point3 col;
			if (i < count3) pblock->GetValue(pb_colors,time,col,ivalid,i);				
			else col = Point3(0,0,0);
			colTab.Append(1,&VR::Vector(col.x, col.y, col.z),500);			
		}
	}

	// Convert tabs to arrays
	pointCount = posTab.Count();

	position = new VR::Vector[pointCount]; 	
	for (int i=0;i<pointCount;i++)
		position[i] = posTab[i];
	posTab.SetCount(0); // Free memory

	if (useRadii) {
		radii = new VR::Ireal[pointCount];
		for (int i=0;i<pointCount;i++)
			radii[i] = radTab[i];
		radTab.SetCount(0); // Free memory		
	}	

	if (useColors) {
		colors = new VR::Vector[pointCount]; 		
		for (int i=0;i<pointCount;i++)
			colors[i] = colTab[i];
		colTab.SetCount(0); // Free memory	
	}
}

void readPointCloud::read(bool useRadii, bool useColors, int &pointCount, VR::Vector* &position, VR::Vector* &velocity, VR::Ireal* &radii, VR::Ireal* &radiiChange, VR::Vector* &colors, IParamBlock2 *pblock, TimeValue time[2]) {
	this->useRadii = useRadii;

	// Start parsing nodes
	int nodeCount = pblock->Count(pb_nodes);
	for (int i=0; i<nodeCount; i++) {
		INode *node;
		pblock->GetValue(pb_nodes,time[0],node,Interval(time[0],time[0]),i);
		if (!node) continue; // No node skip

		// Mesh
		if (readTriMesh(node, time))
			continue;

		// Particles
		ObjectState tos =  node->EvalWorldState(time[0],TRUE);
		if (tos.obj->IsParticleSystem()) {				
			// Simple particle system
			if (readSimpleParticle(node, time))
				continue;	
			// Extended particle system			
			/*if (usePFlowSpawnFix) {
				if (readExtParticleVel(node, pblock, time)) // Using the fix method (using velocity and no changing radius)
					continue;	
			} else {*/
				if (readExtParticle(node, pblock, time)) // Using standard method (using position and include changing radius)
					continue;	
			/*}*/
		}		

		// TODO: Add support for more objects here
	}

	// Extra metaball points, the Point3 array which can be used through maxscript for example
	int count = pblock->Count(pb_points);
	int count2 = pblock->Count(pb_pointSizes);
	int count3 = pblock->Count(pb_colors);
	Interval ivalid = Interval(time[0],time[0]); // This doesn't matter, we would get the interval where the position is valid through it but we don't need it
	for (int i=0; i<count; i++) {
		Point3 pos;
		pblock->GetValue(pb_points,time[0],pos,ivalid,i);
		posTab.Append(1,&VR::Vector(pos.x, pos.y, pos.z),500);
		Point3 pos2;
		pblock->GetValue(pb_points,time[1],pos2,ivalid,i);
		pos = pos2 - pos;
		velTab.Append(1,&VR::Vector(pos.x, pos.y, pos.z),500);			
		if (useRadii) {		
			if (i < count2) {
				float size;
				pblock->GetValue(pb_pointSizes,time[0],size,ivalid,i);
				radTab.Append(1,&size,500);
				float size2;
				pblock->GetValue(pb_pointSizes,time[1],size2,ivalid,i);
				size = size2 - size;
				chaTab.Append(1,&size,500);
			} else {
				radTab.Append(1,&defaultRadius,500);
				chaTab.Append(1,&zero,500);
			}
		}
		if (useColors) {		
			Point3 col;
			if (i < count3) pblock->GetValue(pb_colors,time[0],col,ivalid,i);				
			else col = Point3(0,0,0);
			colTab.Append(1,&VR::Vector(col.x, col.y, col.z),500);			
		}
	}

	// Convert tabs to arrays
	pointCount = posTab.Count();

	position = new VR::Vector[pointCount]; 	
	for (int i=0;i<pointCount;i++)
		position[i] = posTab[i];
	posTab.SetCount(0); // Free memory

	velocity = new VR::Vector[pointCount];
	for (int i=0;i<pointCount;i++)
		velocity[i] = velTab[i];
	velTab.SetCount(0); // Free memory

	if (useRadii) {
		radii = new VR::Ireal[pointCount];
		for (int i=0;i<pointCount;i++)
			radii[i] = radTab[i];
		radTab.SetCount(0); // Free memory
		
		radiiChange = new VR::Ireal[pointCount];
		for (int i=0;i<pointCount;i++)
			radiiChange[i] = chaTab[i];
		chaTab.SetCount(0); // Free memory				
	}

	if (useColors) {
		colors = new VR::Vector[pointCount]; 		
		for (int i=0;i<pointCount;i++)
			colors[i] = colTab[i];
		colTab.SetCount(0); // Free memory	
	}
}